/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/***
 * \file: gpio.c
 *
 * \brief: This file contains the GPIO handler functions to switch
 *         particular GPIO line of respective bank.
 *
 */
#include "gpio.h"
#include "dm8127_platform.h"

/***
 * \brief : GPIO bank 0 pin control function.
 *
 * \b \Descr : This function takes the pin number for GPIO bank zero
 * \n          and it will make it HIGH or LOW as per 2nd parameter of
 * \n          the function.
 *
 * \param : PIN number [IN] pin that has to be changed.
 * \n       pin val    [IN] new value for a gien pin
 *
 * \return: Always true.
 */
int set_gpio_bank0
(
	int gpio,
	int value
)
{
	unsigned int regval;
	static int initialised = 0;

	if (!initialised) {
	/* Reset GPIO Subsystem */
		//GPIO1_SYSCONFIG = 0x00000020; /* Software Reset */
		GPIO0_SYSCONFIG = 0x00000002; /* Software Reset */
		delay(0x90000);
//		GPIO1_SYSCONFIG = 0x100; /* no-idle */
		GPIO0_SYSCONFIG = 0x08; /* no-idle */
		initialised = 1;
	}

	/* Output Enable GPIO16 in Bank 2 */
	regval = GPIO0_OE;
	regval = regval & ~(1 << gpio);
	GPIO0_OE = regval;

	if (!value) {
		/* Set WLAN_ENABLE Low */
		regval = GPIO0_DATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO0_DATAOUT = regval;
		regval = GPIO0_SETDATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO0_SETDATAOUT = regval;
	} else {
		/* Set BT_ENABLE High */
		GPIO0_DATAOUT |= (1 <<  gpio);
		GPIO0_SETDATAOUT |= (1 << gpio);
	}

	return 1;
}


/***
 * \brief : GPIO bank 1 pin control function.
 *
 * \b \Descr : This function takes the pin number for GPIO bank One
 * \n          and it will make it HIGH or LOW as per 2nd parameter of
 * \n          the function.
 *
 * \param : pin number [IN] pin that has to be changed.
 * \n       pin val    [IN] new value for a gien pin
 *
 * \return: Always true.
 */

int set_gpio_bank1
(
	int gpio,
	int value
)
{
	unsigned int regval;
	static int initialised = 0;
#if 1
	if (!initialised) {
	/* Reset GPIO Subsystem */
		//GPIO1_SYSCONFIG = 0x00000020; /* Software Reset */
		GPIO1_SYSCONFIG = 0x00000002; /* Software Reset */
		delay(0x90000);
//		GPIO1_SYSCONFIG = 0x100; /* no-idle */
		GPIO1_SYSCONFIG = 0x08; /* no-idle */
		initialised = 1;
	}
#endif
	/* Output Enable GPIO16 in Bank 2 */
	regval = GPIO1_OE;
	regval = regval & ~(1 << gpio);
	GPIO1_OE = regval;

	if (!value) {
		/* Set WLAN_ENABLE Low */
		regval = GPIO1_DATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO1_DATAOUT = regval;
		regval = GPIO1_SETDATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO1_SETDATAOUT = regval;
	} else {
		/* Set BT_ENABLE High */
		GPIO1_DATAOUT |= (1 <<  gpio);
		GPIO1_SETDATAOUT |= (1 << gpio);
	}

	return 1;
}

/***
 * \brief : Delay routine
 *
 * \b \Descr : This routine give the delay into the calling routine
 * \n          based on the input as a parameter.
 *
 * \param : s32delay [IN] delay count.
 *
 * \return: VOID
 */

int set_gpio_bank2 (int gpio, int value)
{
	unsigned int regval;
	static int initialised = 0;

	if (!initialised) {
	/* Reset GPIO Subsystem */
		GPIO2_SYSCONFIG = 0x00000020; /* Software Reset */
		delay(0x90000);
		GPIO2_SYSCONFIG = 0x08; /* no-idle */
		initialised = 1;
	}

	/* Output Enable GPIO16 in Bank 1 */
	regval = GPIO2_OE;
	regval = regval & ~(1 << gpio);
	GPIO2_OE = regval;

	if (!value) {
		/* Set Low */
		regval = GPIO2_DATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO2_DATAOUT = regval;
		regval = GPIO2_SETDATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO2_SETDATAOUT = regval;
	} else {
		/* Set  High */
		GPIO2_DATAOUT |= (1 <<  gpio);
		GPIO2_SETDATAOUT |= (1 << gpio);
	}

	return 0;
}

/***
 * \brief : GPIO bank 3 pin control function.
 *
 * \b \Descr : This function takes the pin number for GPIO bank zero
 * \n          and it will make it HIGH or LOW as per 2nd parameter of
 * \n          the function.
 *
 * \param : PIN number [IN] pin that has to be changed.
 *          pin val    [IN] new value for a gien pin
 *
 * \return: Always true.
 */

int set_gpio_bank3
(
	int gpio,
	int value
)
{
 	unsigned int regval;
 	static int initialised = 0;

 	if (!initialised) {
 	/* Reset GPIO Subsystem */
 		GPIO3_SYSCONFIG = 0x00000020; /* Software Reset */
 		delay(0x90000);
 		GPIO3_SYSCONFIG = 0x100; /* no-idle */
 		initialised = 1;
 	}

 	/* Output Enable GPIO16 in Bank 2 */
 	regval = GPIO3_OE;
 	regval = regval & ~(1 << gpio);
 	GPIO3_OE = regval;

 	if (!value) {
 		/* Set WLAN_ENABLE Low */
 		regval = GPIO3_DATAOUT;
 		regval = regval & ~(1 << gpio);
 		GPIO3_DATAOUT = regval;
 		regval = GPIO3_SETDATAOUT;
 		regval = regval & ~(1 << gpio);
 		GPIO3_SETDATAOUT = regval;
 	} else {
 		/* Set BT_ENABLE High */
 		GPIO3_DATAOUT |= (1 <<  gpio);
 		GPIO3_SETDATAOUT |= (1 << gpio);
 	}

 	return 1;
 }
